//___________________________________
//                                   \
// Teleporthica MapChange Engine.  version 1.3	$Id: teleporthica.js 108 2008-07-18 21:23:47Z nilton $
//___________________________________/
/*
	To change to a map, when you want several entry points is impossible with ChangeMap() only.
	You may only need this:

		RequireScript("teleporthica.js");

	And then you can use:

		WarpXY(rmpfilename,x,y,tonewlayer);	//(x,y) in Pixels
	or
		WarpTo(rmpfilename,tx,ty,tonewlayer);	//(tx,ty) in Tiles

	The values of x, y and layer will be taken from the EntryPoint if ommited.
	But to get more functionality, you need to set up some things.


	*** SHORTWRITES ***

	Shortwrite the Warp() and WarpXY() functions, now they are easy to call, 
	these functions are automatically created for you:

		Warp = function(r,p,t,r,i,f) {return MapChange.warp(r,p,t,r,i,f);}
		WarpXY = function(r,x,y,l,t,r,i,f) {return MapChange.warpXY(r,x,y,l,t,r,i,f);}
		WarpTo = function(r,tx,ty,l,t,r,i,f) {return MapChange.warpTo(r,tx,ty,l,t,r,i,f);}
		WarpBack = function(retrigger,msg,run,fade) {return MapChange.warpBack(retrigger,msg,run,fade);}
		SeamlessWarpXY = function(new_map,new_XX,new_YY) {MapChange.SeamlessWarpXY(new_map,new_XX,new_YY);}

	(For the parameters you can use, see the real functions below in the code)

	*** WARP points ***

	Inside the Entry script of a map (or in the mapEnter function, if using BindScript), 
	set the mapname and WARP points:

		WARP = { 
			mapname: "How did I end up here?", // The name of the map
 			NorthernEntry : {
				tx: 52,
				ty: 21,
  				run: "ResetPuzzle(1)",
				face: COMMAND_FACE_NORTH,
				name: "Alternative map name" // Overrides mapname
			},  
 			StairsDownToCatacombs: { tx:46, ty:22, dx:8 },
 			Upstairs: { x:162, y:230, layer:2}
 			GoingUp: { tx:null, ty:-1 }
		}


	where tx and ty are tile coordinates (not pixels!), x and y also exist, for the ones that 
	want pixelperfect placement. dx and dy exist to offset by pixels. If the warppoint does not exist, 
	then Warp() will default to the entrypoint. The layer has to be specified only if you need to 
	warp to another layer than the entrypoint has. run is for when you also need to execute something 
	complex like a sound. face runs QueuePersonCommand(person,<face>, true); to face a direction.
	Each value can be ommited, in that case, the defaults will be taken from the entrypoint.
	if tx, x, ty, y or layer is undefined, then that value will be taken from the entrypoint.
	if tx, x, ty, y or layer is null, then that value will be taken from the previous map.
	if tx, x, ty, y or layer are -1, then that maximum possible value will be used (minus 1).

	Then, at a place in any map where you want an exitpoint, define a trigger with  
	Warp(rmpfilename,warppoint,titlemsg,run,wbinfo,fade) for example:

		Warp('mymap.rmp','NorthernEntry');

	Or, you can also use the classical:

		WarpXY(rmpfilename,x,y,tonewlayer,titlemsg,run,warpbackInfoObject)

	These (x,y) are in pixels, use WarpTo() when specifying tiles (the function has 
	the same order of parameters).

		WarpTo(rmpfilename,tx,ty,tonewlayer,titlemsg,run,warpbackInfoObject)

	In the end, everything is optional and you can only use: WarpTo('level2.rmp',1,10);

	titlemsg will override any set mapname
	run Additional command you want to run, after the map change and the repositioning of the input person and after the run defined in the warppoint.
	wbinfo Is the WarpBack info, I'll explain that in a moment
	fade during this warp, use this fade.

	*** Special warp functions ***

	If you need to fake you're entering the map you are currently in (to reset the puzzles on 
	the map, for example) then use:

		MapChange.warpRestart();

	If you need to warp back to the map, just before you came to the current map 
	(for example: to return from a cutscene), use:

		WarpBack();

	If you are unsure if the WarpBack() will warp you back to a place where another trigger 
	will warp you, use:

		WarpBack(true)

	WarpBack will warp you back to the previous map/place, but dont use it unless its a room with no other warps out. 
	WarpBack is useful for a hall with lots of doors that warp you to the same empty room (you make one single map of an empty room). 
	This way, each time you return to the correct door in the hall.
	In order to WarpBack() correctly, you need to give the previous warp with WarpBack Information.

		A warpbackInfoObject can contain the following information:

		x:   When using warpback, it will warp to this x (in pixels)
		dx:  When using warpback, it will offset dx pixels
		tx:  When using warpback, it will warp to this x (in tiles)
		dtx: When using warpback, it will offset dtx tiles (it automatically centers the sprite on that tile)
                y:   When using warpback, it will warp to this y (in pixels)
                dy:  When using warpback, it will offset dy pixels
                ty:  When using warpback, it will warp to this y (in tiles)
                dty: When using warpback, it will offset dty tiles (it automatically centers the sprite on that tile)
		layer: Layer 
		dir: Facing direction of the input person
		name: Name of the input person
		rmp: Name of the previous map


	So, lets say we have a door and we enter into it going up, and that when we warp back, we dont want to be standing on that trigger again.
	We want us to warp back to one tile down, so dty=1

		Warp('emptyroom.rmp', 'EmptyRoom', 'An empty room', {dty:1} )

	Then, from the empty room we exit with:

		WarpBack();

	note: You can use more values together in the warpbackInfoObject, just separate them with ",".
	example: warp to Y tile 5 (ty=4 + dty=1). and X 338 pixels (x=340 + dx=-2)

		Warp('emptyroom.rmp', 'EmptyRoom', 'An empty room', {dty:1, x:340,dx:-2,ty:4} );

	
	run this when you just entered a dungeon:

		MapChange.storeWarpBackInfo();

	Then, when you need to immediately exit the dungeon, just 
		MapChange.restoreWarpBackInfo(); MapChange.warpBack(true);
	You could also manage your own warpback point, or use these functions to restore to a savespot 
	(note that maybe you'll want a mix of loading a savegame, but keeping all experience)



	MapChange.SeamlessWarpXY() is a very special warp type, it allows to move to another map in a 
	zelda-link-to-the-past style. It works better on maps that are of equal size.
	You call it inside the map SCRIPT_ON_LEAVE_MAP_* scripts like so:
	NORTH: SeamlessWarpXY("map.rmp", null, -1);
	EAST:  SeamlessWarpXY("map.rmp", 0, null);
	SOUTH: SeamlessWarpXY("map.rmp", null, 0);
	WEST:  SeamlessWarpXY("map.rmp", -1, null);
	
	Note that a screenshot is taken from both maps, so to get rid of the statusbar (if you have one)
	You may want to clear the renderscript/set the renderscript before running this, or even better,
	put this in your renderscript code:  if(!MapChange.active) { <Do your statbar things here>}
	You can still draw the statusbar while scrolling to the next map, just redefine .scrollUpdateAndDraw()
	Also, make something obstruct at the edge of 4 maps, or bad things will happen.


	*** Tweaking ***

	You can redefine these functions:
	.preWarp() is everything you need to do before warping, after fading out. 
	Just like .postWarp() is what you redefine after a warp, before fading in.
	.showName() is what you redefine after a warp, after fading in.
	MapChange.active is 0 when not active, 2 when fading out, and 1 just before fading in. (and 3 when seamless-ing)

	To change the fade-in speed from the current fade to 50, just: code somewhere MapChange.fadeInFrames=50;
	You can also change the way the current fade is done by directly redefining MapChange.fadeIn();
	The best way to create fade is to define your own, at the end of this file you can see
	MapChange.DefineFade('grayfade', .... );
	Note that fadeIn() and fadeOut() only get called once, from there overload UpdateAndDraw() if you need to loop.
	
	For example, I've already coded spotfade, so to activate it do this:

	MapChange.setFade('spot');


	*** FLAGS ***

	MapChange.maps holds information about visited maps, you may want to use/save this information.
	To make our life easier, define a global variable like so:

		var FLAGS = {MAP:MapChange.maps};

	If you use Spheritype, maybe even this:

		var FLAGS = { MAP: MapChange.maps, STATE: ScriptBind.world.state, MAPS: ScriptBind.world.maps };

	This object can hold events that have happened. Do not confuse with EventMarker[], which can be resetted each time a cutscene starts.
	FLAGS.MAP will hold flags of events that happen inside a certain map. We can check things like: 
		if(FLAGS.MAP[GetCurrentMap()].hasgrabbedtheloot) {...};
	Be scarse with those objects, as they are all loaded/saved. Note that you can also use: 
		if(FLAGS.myevent1==true){...};

	Flags are usefull when we have a one-time treasure or cutscene that playes only once, we can mark it inside this object with a persistant flag.

		function hasHappened(it,val){ if(val==undefined) return FLAGS[it]; return FLAGS[it]=val; }


	*** DO NOT CREATE CIRCULAR WARPPOINTS ***

	Its bad when you warp on top of a trigger that warps you elsewhere, it will trigger this warp
	and you'll probably be back on the same place you triggered from, or worse, stuck being 
	triggered back and forth between triggers. The way to avoid it is create a trigger at tile (X,Y)
	and return at tile (X,Y+1) when you would be walking south ( (X+1,Y)) when walking east, etc.)
	If you do need circular warppoints, increase this variable like so:  MapChange.delaybetweenwarps=300;

*/

var MapChange = new TeleporthicaEngine("MapChange");
var WARP = {}; // Hold warp point information (cleared after each mapchange. And may be redefined in each map entry script)


// Short-write some functions, passing all parameters verbatim to the real function.
Warp = function() {return MapChange.warp.apply(MapChange, arguments);}
WarpXY = function() {return MapChange.warpXY.apply(MapChange, arguments);}
WarpTo = function() {return MapChange.warpTo.apply(MapChange, arguments);}
WarpBack = function() {return MapChange.warpBack.apply(MapChange, arguments);}
SeamlessWarpXY = function() {return MapChange.SeamlessWarpXY.apply(MapChange, arguments);}


//-----------------------------------------------------------------------------------//

/*
 * Create a new Teleporthica Object
 * @constructor
 * 
 * @param {string} ObjectName Mandatory must be the same as the created variable.
 * @param {string} InputPersonName optional Input Person (is autoassigned, and can also be set with MapChange.setInputPerson() )
 * @return A new TeleporthicaEngine Object
 */
function TeleporthicaEngine(objname,GIP)
{
	if(this instanceof TeleporthicaEngine == false) {
		return new TeleporthicaEngine(objname,GIP);
	}
	this.objname=objname;
	this.maps={}; //Holds persistent information about flags and visited maps
	this.previous= {}; //previous map information
	this.stored= {}; //Used with store/restore WarpBack
	this.current = {rmp:""}; //Holds the information of the current map
	this.entryPoint = {X:0,Y:0,L:0}; //Holds the information of the current map's entrypoint
	this.next_warp=0; //When warping onto a warp trigger, dont execute it. (use warpNow() if you DO want that)
	this.active=false; //is warp in progress
	this.warpid=""; // the warpid, how you got onto a map
	this.oldwarpid="";
	this.delaybetweenwarps=100; //milliseconds delays between warps (prevents multiple mapchange triggers from happening)
	this.fps=60; //default Frames Per Second.
	this.centeredWarpBack = true; //must person be centered on tile when warping back?
	this.FADES=new Array(); //Array to hold different kind of fades, we define a few here
	//Define fade: No fade at all. enable it like this: MapChange.setFade(0);
	this.FADES[0] = {
		fadeInFrames : 0,
		fadeOutFrames: 0,
		fadeIn : function(){},
		fadeOut: function(){}
	};
	//Fade to Black, start map in blackness
	this.FADES['toblack'] = {
		fadeInFrames : 0,
		fadeOutFrames: 10,
		fadeIn : function(){ SetColorMask(CreateColor(0, 0, 0, 255), 0, 'setblack'); },
		fadeOut: function(){ SetColorMask(CreateColor(0, 0, 0, 255), this.FADES['toblack'].fadeOutFrames, 'toblackOUT'); }
	};
	//Default fade: Fade to black and fade from black.
	this.FADES['fade'] = {
		fadeInFrames : 10,
		fadeOutFrames: 10,
		fadeIn : function(){ SetColorMask(CreateColor(0, 0, 0, 0), this.FADES['fade'].fadeInFrames, 'fadeIN'); },
		fadeOut: function(){ SetColorMask(CreateColor(0, 0, 0, 255), this.FADES['fade'].fadeOutFrames, 'fadeOUT'); }
	};
	// Start from blackness, do normal fadein
	this.FADES['fromblack'] = {
		fadeInFrames : 10,
		fadeOutFrames: 0,
		fadeIn : function(){ SetColorMask(CreateColor(0, 0, 0, 255), 0, 'fromblackIN1'); SetColorMask(CreateColor(0, 0, 0, 0), this.FADES['fromblack'].fadeInFrames, 'fromblackIN2'); },
		fadeOut: function(){ SetColorMask(CreateColor(0, 0, 0, 255), 0, 'ffromblackOUT'); }
	};

	//Define some defaults
	this.fadeIn = this.FADES['fade'].fadeIn;
	this.fadeOut= this.FADES['fade'].fadeOut;
	this.fadeInFrames = this.FADES['fade'].fadeInFrames;
	this.fadeOutFrames= this.FADES['fade'].fadeOutFrames;
	this.fadeName='fade';
	this.fadeName_="";

	//User definable variables.
	this.showName = function(){}; //Redefine this function to display the mapname at entry.
	this.postWarp = function(){}; //userdefinable. called just after ChangeMap() and repos(); note, you still can use SetDefaultMapScript()
	this.preWarp = function(){}; //userdefinable. called just before ChangeMap() and repos();
	//this.mapInit= function(){}; //userdefinable. Could replace SetDefaultMapScript(SCRIPT_ON_ENTER_MAP,"YourFunctionHere"). Better use that one.

	this.GIP = GIP? GIP:IsInputAttached() ? GetInputPerson() : ""; // The input person

	this.TileToMapX = function(x){if(x<0)x=0; return x=(x<<4)+7;} //Works only if tiles are 16x16, see Phenibut for alternative.
	this.TileToMapY = function(y){if(y<0)y=0; return y=(y<<4)+7;} //+7 is offset to center the sprite on the tile, you may need to adjust it.
	this.MapToTileX = function(x){if(x<0)x=0; return x>>4;}
	this.MapToTileY = function(y){if(y<0)y=0; return y>>4;}

	this.TileToMapX0 = function(x){return x=(x<<4);} // Used for calculating offsets
	this.TileToMapY0 = function(y){return y=(y<<4);} // Used for calculating offsets

	this.TileToMapXWrap = function(x){if(x<0)x=GetLayerWidth(this.current.layer)-x; return x=(x<<4)+7;} //Works only if tiles are 16x16, see phenibut.js for alternative.
	this.TileToMapYWrap = function(y){if(y<0)y=GetLayerWidth(this.current.layer)-y; return y=(y<<4)+7;} //+7 is offset to center the sprite on the tile, you may need to adjust it.
	this.MapToTileXWrap = function(x){if(x<0)x=GetLayerWidth(this.current.layer)*GetTileWidth()-x; return x>>4;}
	this.MapToTileYWrap = function(y){if(y<0)y=GetLayerHeight(this.current.layer)*GetTileHeight()-y; return y>>4;}

	this.scrolldelay= 5; //Delay in ms for seamless scroll (0: no delay, scrolls as fast as your CPU can handle. )
	this.scrollskip= 16; //scroll skip for seamless scroll (0: very smooth, very slow, GetScreenWidth(): too fast to see anything )
	this.scrollUpdateAndDraw=function(){}; //want to do something while scrolling to the other map? (like a statusbar?)
	this.UpdateAndDraw=function(){}; //Helper function
}

/**
 * Sets the person on which all teleportation will be applied to by default.
 * If you start teleporting to maps while the input is not attached, you need to set this (once).
 * @param {string} person Input person name. If left empty, will default to GetInputPerson()
 * @returns the name of the chosen input person
 */
TeleporthicaEngine.prototype.setInputPerson = function(person) {
	return this.GIP=person||(IsInputAttached()?GetInputPerson():this.GIP);
}
	     
/**
 * Internal Function. Will store information about the position of the Input person just after warping to a new map.
 * These values are held inside the object MapChange.current. It stores: 
 * rmp The name of the current map
 * x the position on the map
 * y the position on the map
 * layer the layer on the map
 * direction the facing direction
 * name The name of the input person
 * warppoint The name of the warppoint
 * Redefine it if you need to store more properties of the input person.
 * @param warppoint Optional name of the last warppoint called
 * @returns false if the Input Person is not defined (will only set the warppoint, if defined), or true if all data could be stored
 */
TeleporthicaEngine.prototype.updateCurrentInfo = function(warppoint){ 
	if (!this.GIP){
		if(warppoint)
			this.current={warppoint:warppoint};
		else
			this.current={};
		return false;
	}
	this.current={
		rmp:GetCurrentMap(),
		x:GetPersonX(this.GIP),
		y:GetPersonY(this.GIP),
		layer:GetPersonLayer(this.GIP),
		direction:GetPersonDirection(this.GIP),
		name:this.GIP,
		warppoint:warppoint
	};
	return true;
}

/**
 * Internal Function. Will store information about the position of the Input person just before warping to a new map.
 * @param {object} wbinfo Warp Back information, to redefine the position you warpBack() to.
 * @returns false if the Input Person is not defined (will only set the warppoint, if defined), or true if all data could be stored
 */
//wbinfo {x dx tx dtx y dy ty dty layer rmp name dir}
TeleporthicaEngine.prototype.updatePreviousInfo = function(wbinfo){ 
	if (!this.GIP){
		this.previous={
			rmp: wbinfo && wbinfo.rmp ? wbinfo.rmp : GetCurrentMap(),
			name: wbinfo && wbinfo.name ? wbinfo.name : this.GIP,
			warppoint: wbinfo && wbinfo.warppoint ? wbinfo.warppoint : ""
		}
		return false;
	};
	if(wbinfo){
		wbinfo.x = wbinfo.x? wbinfo.x: wbinfo.tx? this.TileToMapX(wbinfo.tx) : GetPersonX(this.GIP);
		if(!wbinfo.dx) wbinfo.dx =0;
		if(wbinfo.dtx)wbinfo.dx += this.TileToMapX0(wbinfo.dtx);

		wbinfo.y= wbinfo.y? wbinfo.y: wbinfo.ty? this.TileToMapY(wbinfo.ty) : GetPersonY(this.GIP);
		if(!wbinfo.dy) wbinfo.dy =0;
		if(wbinfo.dty)wbinfo.dy += this.TileToMapY0(wbinfo.dty);

		wbinfo.layer= wbinfo.layer? wbinfo.layer:GetPersonLayer(this.GIP);

		this.previous={
			rmp:wbinfo.rmp||GetCurrentMap(),
			x:wbinfo.x,
			dx:wbinfo.dx||0,
			y:wbinfo.y,
			dy:wbinfo.dy||0,
			layer:wbinfo.layer,
			direction:wbinfo.dir||GetPersonDirection(this.GIP),
			name:wbinfo.name||this.GIP,
			warppoint:wbinfo.warppoint
		};
	} else
		this.previous={
			rmp:GetCurrentMap(),
			x:GetPersonX(this.GIP),
			y:GetPersonY(this.GIP),
			layer:GetPersonLayer(this.GIP),
			direction:GetPersonDirection(this.GIP),
			name:this.GIP
		};
	return true;
}
/**
 * Will store the information set with updatePreviousInfo() to a secure place, so we can get it back
 * Run this when you just entered a dungeon, Then, when you need to immediately exit the dungeon, just call 
 * {@link TeleporthicaEngine#restoreWarpBackInfo} and MapChange.warpBack(true);
 * TODO: use clone
 */
TeleporthicaEngine.prototype.storeWarpBackInfo = function(){ 
	this.stored = this.previous;
}

/**
 * Will restore the information of MapChange.previous with information stored by {@link TeleporthicaEngine#storeWarpBackInfo}
 */
TeleporthicaEngine.prototype.restoreWarpBackInfo = function(){ 
	this.previous = this.stored;
}

/**
 * Internal Function. Sets the x, y and layer of the entry point.
 * Will not work correctly if you move the input person with a defined SetDefaultMapScript()
 * @returns false if there is no input person to read the entrypoint information from, true if there is.
 */
TeleporthicaEngine.prototype.updateEntryPointInfo= function(){ 
	this.entryPoint.X=0;
	this.entryPoint.Y=0;
	this.entryPoint.L=0;
	this.setVisited();
	this.updateLayerObject();
	if(!this.GIP)return false;
	this.entryPoint.X=GetPersonX(this.GIP);
	this.entryPoint.Y=GetPersonY(this.GIP);
	this.entryPoint.L=GetPersonLayer(this.GIP);
	return true;
}

/**
 * Internal Function. Makes an object so we can get a layer index by name (gets automatically updated for each map)
 * You can use it like so: function GetLayerIndex(name){ return MapChange.LayerObject[name]; }
 */
TeleporthicaEngine.prototype.updateLayerObject= function(){
	var i=GetNumLayers()-1;
	this.LayerObject = {};
	do{
		this.LayerObject[GetLayerName(i)]=i;
	}while(i--);
}

/**
 * Flags the current map as visited (called automatically)
 * Now, inside a map we can check things like: if(MapChange.maps[GetCurrentMap()].visited) {...};
 * @param {string} rmp Name of the map, leave undefined to get the current map.
 * @param {integer} times Number of times the map has been visited. Leave undefined to autoincrease this number by one.
 * @returns the number of times the map has been visited
 */
TeleporthicaEngine.prototype.setVisited= function(rmp,times){ 
	rmp=rmp||GetCurrentMap();
	if ( !this.maps[rmp] )
		this.maps[rmp] = new Object();
	if (times)
		return this.maps[rmp].visited = times;
	else 
		return ++this.maps[rmp].visited;
}

/**
 * Starts a map, like MapEngine()
 * @param {integer} fps Default Frames Per Second. The default is 60. 
 * @param {string} rmpfilename Map name
 * @param {string} warppoint Warp point you want to teleport to
 * @param {string} titlemsg Optional title message (will override all others
 * @param {string} fade Type of fade (it will only use the fadeIn part)
 * @returns true after the mapengine has died again.
 */
TeleporthicaEngine.prototype.start = function(fps,rmpfilename,warppoint,titlemsg,fade) {
	this.GIP = IsInputAttached()?GetInputPerson():this.GIP;
	this.fps = fps;
	WARP = {};
	this.current.rmp = rmpfilename;
	if(fade !==undefined) this.setNextFade(fade);
	var fof = this.fadeOutFrames;
	//this.fadeOutFrames = 1;
	//this.fadeOutFrames = fof;

	this.UpdateAndDraw = function(){
		this.updateEntryPointInfo();
		this.updatePreviousInfo();
		SetClippingRectangle(0, 0, GetScreenWidth(), GetScreenHeight());
		this.warpNow("",warppoint,titlemsg);
		this.done();
	}; 
	this.setVisited(rmpfilename,0);
	ApplyColorMask(CreateColor(0,0,0,255));
	SetClippingRectangle(0, 0, 0, 0); // The only way to be able to start with a black screen
 	MapEngine(rmpfilename, fps);	//Now start the map engine 
	return true; //Finally, when the mapengine stops, we exit.
}

/**
 * Map Change to another map using a warp point reference
 * for example: MapChange.warp('mymap.rmp', 'SouthCaveEntry'); 
 * Where SouthCaveEntry is defined inside the WARP object, possibly in the map entry script.
 * @param {string} rmpfilename Map filename
 * @param {string} warppoint Warp point you want to teleport to
 * @param {string} titlemsg Optional title message (will override all others)
 * @param {string} run Additional command you want to run, after the map change and the repositioning of the input person and after the run defined in the warppoint
 * @param {object} wbinfo Modifies the warp back information. See {@link TeleporthicaEngine#updatePreviousInfo} for details.
 * @param {string} fade Type of fade (it will only be used for this warp)
 * @returns true if we did warp, false or undefined if we didnt warp.
 */
TeleporthicaEngine.prototype.warp = function(rmpfilename,warppoint,titlemsg,run,wbinfo,fade) {
	if(this.retrigger && this.detrigger())
		return false;
	if((this.next_warp>GetTime()&&(this.oldwarpid==rmpfilename+';'+warppoint))||(this.warpid==rmpfilename+';'+warppoint))
		return false;
	//if(!FileExists(rmpfilename))return false;
	this.GIP = IsInputAttached() ? GetInputPerson() : this.GIP;
	this.updatePreviousInfo(wbinfo);
	if(fade) this.setNextFade(fade);
	this.warpNow(rmpfilename,warppoint,titlemsg,run);
	return true;
}

/**
 * Internal function used by {@link TeleporthicaEngine#warp}.
 * @param {string} rmpfilename Map name
 * @param {string} warppoint Warp point you want to teleport to
 * @param {string} titlemsg Optional title message (will override all others)
 * @param {string} run Additional command you want to run, after the map change and the repositioning of the input person and after the run defined in the warppoint
 */
TeleporthicaEngine.prototype.warpNow = function(rmpfilename,warppoint,titlemsg,run) {
	this.next_warp=GetTime()+this.delaybetweenwarps;
	this.oldwarpid=this.warpid;
	this.warpid=rmpfilename+';'+warppoint;
	this.active=2;
	this.fadeOut();
	this.preWarp();
	if(rmpfilename){
		WARP={};
		this.setVisited(rmpfilename);
		//this.mapInit();
	}

	var str=
		(rmpfilename?
			this.objname+".current.rmp='"+rmpfilename+"';" + 
			"ChangeMap('" + rmpfilename + "');" +
			this.objname+".updateEntryPointInfo();"
			:
			"") +
		this.objname+".repos('"+warppoint+"','"+run+"');" +
		this.objname+".active=1;"+
		this.objname+".postWarp();" +

		this.objname+".fadeIn();" +
		(titlemsg?"WARP.mapname='"+titlemsg+"';":"") +
		this.objname+".showName();"
		+"SetDelayScript("+this.fadeInFrames+",'"+this.objname+".done()');"
		+(this.fadeName_ ? "SetDelayScript("+(this.fadeInFrames+1)+",'"+this.objname+".setFade(\""+this.fadeName_+"\")');" :"" )
	;

	if(this.fadeOutFrames>0)
		SetDelayScript(this.fadeOutFrames, str);
	else
		eval(str);
}

/**
 * Move the input person to the coordinates defined by the warppoint, if the warppoint does not exist, it will default to the entrypoint.
 * @param {string} warppoint Warp point you want to teleport to
 * @param {string} run Additional command you want to run, after the map change and the repositioning of the input person and after the run defined in the warppoint
 * @returns a boolean that tells us if the current information has been updated, undefined if there was no GIP
 */
TeleporthicaEngine.prototype.repos = function(warppoint,run) {
	this.GIP=IsInputAttached()?GetInputPerson():this.GIP;
	if((warppoint!==undefined) && WARP[warppoint]){
		if(WARP[warppoint].name) WARP.mapname=WARP[warppoint].name;

		if(this.GIP){

			var layer=WARP[warppoint].layer;
			if (layer<0)
				layer += GetNumLayers() + 1;
			if(layer === null)
				layer = this.previous.layer;

			if(layer !== undefined)
				SetPersonLayer(this.GIP, layer); 
			else
				layer = GetPersonLayer(this.GIP);

			var X = undefined;
			if( (WARP[warppoint].tx===null) || (WARP[warppoint].x===null) )
				X = this.previous.x;
			else if(WARP[warppoint].tx !== undefined)
				X = WARP[warppoint].tx<0 ? (GetLayerWidth(layer)-WARP[warppoint].tx)*GetTileWidth()-1 : this.TileToMapX(WARP[warppoint].tx);
			else if(WARP[warppoint].x !== undefined)
				X = WARP[warppoint].x<0 ? GetLayerWidth(layer)*GetTileWidth()-WARP[warppoint].x-1 : WARP[warppoint].x;
			if(WARP[warppoint].dx !== undefined)
				X += WARP[warppoint].dx;

			if(X !== undefined)
				SetPersonX(this.GIP, X);

			var Y = undefined;
			if( (WARP[warppoint].ty===null) || (WARP[warppoint].y===null) )
				Y = this.previous.y;
			else if(WARP[warppoint].ty !== undefined)
				Y = WARP[warppoint].ty<0 ? (GetLayerHeight(layer)-WARP[warppoint].ty)*GetTileWidth()-1 : this.TileToMapX(WARP[warppoint].ty);
			else if(WARP[warppoint].y !== undefined)
				Y = WARP[warppoint].y<0 ? GetLayerHeight(layer)*GetTileWidth()-WARP[warppoint].y-1 : WARP[warppoint].y;
			if(WARP[warppoint].dy !== undefined)
				Y += WARP[warppoint].dy;

			if(Y !== undefined)
				SetPersonY(this.GIP, Y);

			if(WARP[warppoint].face)
				QueuePersonCommand(this.GIP,WARP[warppoint].face, true);
		};
		if(WARP[warppoint].run){
			if(typeof WARP[warppoint].run == 'function')
				WARP[warppoint].run();
			else
				try{
					eval(WARP[warppoint].run);
				}catch(e){
					this.HandleError("Teleporthica run error for warppoint '"+warppoint+"':\n",e)
				}
		};
		return this.updateCurrentInfo(warppoint);
	};

	if(run){
		if(typeof run == 'function')
			run();
		else
			try{
				eval(run);
			}catch(e){
				this.HandleError("Teleporthica 'run' parameter evaluation error for warppoint'"+warppoint+"':\n",e)
			}
	};

	// No warppoint defined, default to the Entry Point.
	if(!this.GIP)return;
	SetPersonX(this.GIP,this.entryPoint.X);
	SetPersonY(this.GIP,this.entryPoint.Y);
	SetPersonLayer(this.GIP,this.entryPoint.L);
	return this.updateCurrentInfo(warppoint);
}

TeleporthicaEngine.prototype.HandleError = function(msg,e){
	for (var i in e) msg += i + ' = ' + e[i] +"\n";
	if(this.name) msg = "While '"+this.name+"':\n" + msg;
	Abort(msg);
}


/**
 * Internal function, will stop multiple warps from happening if for some reason the warp gets called multiple times.
 * It lets retrigger run once only.
 * @returns true if it stops the current warp, false if it decides to let the warp continue.
 */
TeleporthicaEngine.prototype.detrigger = function(){
	if(!--this.retrigger){
		if(this.next_warp+this.delaybetweenwarps>GetTime())
			return true;
	}
	return false;
}


/**
 * Map Change to another map using x,y coordinates.
 * for example: MapChange.warpXY('mymap.rmp', 10,15, undefined);
 * @param {string} rmpfilename Map name. Keep undefined if we teleport to the same map.
 * @param {integer} x X coordinate we want to teleport to. use null to use the coordinate we had un the previous map before warping, undefined uses data from entry point
 * @param {integer} y Y coordinate we want to teleport to. use null to use the coordinate we had un the previous map before warping, undefined uses data from entry point
 * @param {integer} tonewlayer To the new layer we want to teleport to. Keep undefined to use the layer of the Entry Point.
 * @param {string} titlemsg Optional title message (will override all others
 * @param {string} run Additional command you want to run, after the map change and the repositioning of the input person.
 * @param {object} wbinfo Modifies the warp back information. See {@link TeleporthicaEngine#updatePreviousInfo} for details.
 * @param {string} fade Type of fade (it will only be used for this warp)
 * @returns true if we did warp, false or undefined if we didnt warp.
 */
TeleporthicaEngine.prototype.warpXY = function(rmpfilename,x,y,tonewlayer,titlemsg,run,wbinfo,fade) {
	if(this.retrigger && this.detrigger())
		return false;
	if((this.next_warp>GetTime()&&(this.oldwarpid==rmpfilename+';'+x+';'+y+';'+tonewlayer))||(this.warpid==rmpfilename+';'+x+';'+y+';'+tonewlayer))
		return false;
	//if(!FileExists(rmpfilename))return false;
	this.GIP=IsInputAttached()?GetInputPerson():this.GIP;
	this.updatePreviousInfo(wbinfo);
	if(x===null)
		x=this.previous.x;
	if(y===null)
		y=this.previous.y;
	if(fade) this.setNextFade(fade);
	this.warpXYNow(rmpfilename,x,y,tonewlayer,titlemsg,run);
	return true;
}


/**
 * Internal function used by {@link TeleporthicaEngine#warpXY}.
 * @param {string} rmpfilename Map name
 * @param {integer} x X coordinate we want to teleport to. use null to use the coordinate we had un the previous map before warping, undefined uses data from entry point
 * @param {integer} y Y coordinate we want to teleport to. use null to use the coordinate we had un the previous map before warping, undefined uses data from entry point
 * @param {integer} tonewlayer To the new layer we want to teleport to. Keep undefined to use the layer of the Entry Point.
 * @param {string} titlemsg Optional title message (will override all others)
 * @param {string} run Additional command you want to run, after the map change and the repositioning of the input person.
 */
TeleporthicaEngine.prototype.warpXYNow = function(rmpfilename,x,y,tonewlayer,titlemsg,run) {
	this.next_warp = GetTime() + this.delaybetweenwarps;
	this.oldwarpid = this.warpid;
	this.warpid = rmpfilename+';'+x+';'+y+';'+tonewlayer;
	this.current.rmp = rmpfilename;
	this.active = 2;
	this.preWarp();
	this.fadeOut();
	if(rmpfilename){
		WARP = {}; 
		this.setVisited(rmpfilename); 
		//this.mapInit();
	}
	var str =
		(rmpfilename?this.objname+".current.rmp='"+rmpfilename+"';ChangeMap('" + rmpfilename + "');"+this.objname+".updateEntryPointInfo();":"") +
		this.objname+".reposXY("+x+","+y+","+tonewlayer+",'"+run+"');" +
		this.objname+".active=1;"+
		this.objname+".postWarp();" +
		this.objname+".fadeIn();" +
		(titlemsg?"WARP.mapname='"+titlemsg+"';":"") +
		this.objname+".showName();"
		+"SetDelayScript("+this.fadeInFrames+",'"+this.objname+".done()');"
		+(this.fadeName_ ? "SetDelayScript("+(this.fadeInFrames+1)+",'"+this.objname+".setFade(\""+this.fadeName_+"\")');" :"" )
	;
	if(this.fadeOutFrames>0)
		SetDelayScript(this.fadeOutFrames, str);
	else
		eval(str);
}

/**
 * Move the input person to the pixel coordinates defined by x,y and layer. 
 * @param {integer} x X coordinate we want to teleport to. Keep undefined to use the coordinate from the entrypoint.
 * @param {integer} y Y coordinate we want to teleport to. Keep undefined to use the coordinate from the entrypoint.
 * @param {integer} layer  Layer we want to teleport to. Keep undefined to use the layer of the Entry Point.
 * @param {string} run Additional command you want to run, after the map change and the repositioning of the input person.
 * @returns a boolean that tells us if the current information has been updated, undefined if there was no GIP
 */
TeleporthicaEngine.prototype.reposXY = function(x,y,layer,run) {
	this.GIP=IsInputAttached()?GetInputPerson():this.GIP;
	if(typeof layer == "number")
		SetPersonLayer(this.GIP, layer);
	layer=GetPersonLayer(this.GIP);
	if(typeof x == "number"){
		if(x<0)
			x=GetLayerWidth(layer)*GetTileWidth()-x-1;
		SetPersonX(this.GIP, x);
	}
	if(typeof y == "number"){
		if(y<0)
			y=GetLayerHeight(layer)*GetTileHeight()-y-1;
		SetPersonY(this.GIP, y);
	}

	if(run){
		if(typeof run == 'function')
			run();
		else
			try{
				eval(run);
			}catch(e){
				this.HandleError("Teleporthica evaluation error:\n",e)
			}
	}
	return this.updateCurrentInfo();
}

/**
 * Map Change to another map, like warpXY, but the x,y coordinates are for tiles, and not pixels.
 * for example: MapChange.warpXY('mymap.rmp', 10,15, undefined);
 * @param {string} rmpfilename Map name. Keep undefined if we teleport to the same map.
 * @param {integer} tileX X coordinate we want to teleport to. Keep undefined to use the coordinate from the Entry Point.
 * @param {integer} tileY Y coordinate we want to teleport to. Keep undefined to use the coordinate from the Entry Point.
 * @param {integer} tonewlayer To the new layer we want to teleport to. Keep undefined to use the layer of the Entry Point.
 * @param {string} titlemsg Optional title message (will override all others)
 * @param {string} run Additional command you want to run, after the map change and the repositioning of the input person.
 * @param {object} wbinfo Modifies the warp back information. See {@link TeleporthicaEngine#updatePreviousInfo} for details.
 * @param {string} fade Type of fade (it will only be used for this warp)
 * @returns true if we did warp, false or undefined if we didnt warp.
 */
TeleporthicaEngine.prototype.warpTo = function(rmpfilename,tileX,tileY,tonewlayer,titlemsg,run,wbinfo,fade) {
	return this.warpXY(rmpfilename,
		tileX==undefined?undefined: tileX<0?-1:this.TileToMapX(tileX),
		tileY==undefined?undefined: tileY<0?-1:this.TileToMapY(tileY),
		tonewlayer,
		titlemsg,
		run,
		wbinfo,
		fade
	);
}

/**
 * warp back to the previous map. If you land on a trigger that executes another warp, you can optionally disable it.
 * Usually, you use warpBack to warp you back to the previous teleport, and let it trigger a warp that will teleport you to your current map,
 * so its like you just entered it. Useful to, for example, reset all the puzzels in the current map. But you may have modified the warp back
 * information (see wbinfo), in that case, it will just return to the previous map.
 * @param {boolean} retrigger True if we want to warpBack, and execute the trigger we are standing on.
 * @param {string} msg Title message (optional)
 * @param {string} run Additional command you want to run, after the map change and the repositioning of the input person.
 * @param {string} fade Type of fade (it will only be used for this warp)
 * @returns 0 if Teleporthica was already active, 1 if we warped back.
 * Note: there is a global option, centeredWarpBack, which will recenter the person back on the tile, true by default.
 */
TeleporthicaEngine.prototype.warpBack = function(retrigger,msg,run,fade) {
	if(this.active)
		return 0;
	if(retrigger)
		this.retrigger=2;
	if(fade) this.setNextFade(fade);
	if(this.centeredWarpBack) {
		this.warpXY(this.previous.rmp,
			this.TileToMapX(this.MapToTileX(this.previous.x)) + (this.previous.dx||0),
			this.TileToMapY(this.MapToTileY(this.previous.y)) + (this.previous.dy||0),
			this.previous.layer,
			msg,
			run
		);
	}else{
		this.warpXY(this.previous.rmp,
			this.previous.x + (this.previous.dx||0),
			this.previous.y + (this.previous.dy||0),
			this.previous.layer,
			msg,
			run
		);
	}
	return 1;
}

/**
 * Called usually to manually reset the puzzels in a room, much faster and reliable than warpBack. This function does not use WARP{} information, so
 * if you had something defined in run, you need to specify it here separately.
 * @param {string} msg Title message (optional)
 * @param {string} run Additional command you want to run, after the map change and the repositioning of the input person.
 * @param {string} fade Type of fade (it will only be used for this warp)
 */
TeleporthicaEngine.prototype.warpRestart = function(msg,run,fade) {
	if(this.active)
		return 0;
	if(fade) this.setNextFade(fade);
	if(this.current.warppoint)
		this.warpNow(this.current.rmp,this.current.warppoint,msg);
	else
		this.warpXYNow(this.current.rmp,this.current.x,this.current.y,this.current.layer,msg,run);
	return 1;
}

//-----------------------------------------------------------------------------------//
/**
 * Resets and clears Teleporthica
 * @param {string} newGIP Sets a new input person, if zero or undefined, it will try to get the input person, else stick to the old value.
 * @param {Boolean} clearMaps Clear visited maps information
 * @param {Boolean} clearFunctions Clears preWarp() and postWarp() user defined functions.
 */
TeleporthicaEngine.prototype.reset=function(newGIP,clearMaps,clearFunctions){
	if(clearMaps)
		this.maps = {};
	if(clearFunctions) {
		//this.mapInit = function(){};
		this.preWarp = function(){};
		this.postWarp = function(){};
	}
	this.UpdateAndDraw = function(){};
	this.active = false;
	this.waiting = false;
	this.GIP = typeof newGIP=='string'? newGIP : IsInputAttached() ? GetInputPerson() : this.GIP;
	//this.updatePreviousInfo();
	this.updateCurrentInfo();
	GarbageCollect();
}

/**
 * Forces a clear of the UpdateAndDraw teleporthica function.
 */
TeleporthicaEngine.prototype.done = function(){
	this.UpdateAndDraw = function(){};
	this.active = false;
}


//-----------------------------------------------------------------------------------//

/*
 * 
 */
TeleporthicaEngine.prototype.SeamlessWarpXY=function(new_map,new_XX,new_YY){
	//Only allow 1 warp at a time (mapscripts tend to trigger multiple times)
	if(this.active)
		return 0;
	else
		this.active=3;

	//Save the original functions, the user may have modified them.
	this.preWarp_=this.preWarp; 
	this.postWarp_=this.postWarp; 

	//.preWarp :Copy current map
	this.preWarp = function(){
		this.preWarp_(); //Run original preWarp()
		SetColorMask(CreateColor(255, 255, 255, 0),0); //Remove the blackfadeout from the mapengine (will be overwritten soon by fadein anyways.
		//this.render=GetRenderScript();
		//SetRenderScript(""); //Get rid of statusbar and other fx, before rendering the map
		this.GIP=IsInputAttached()?GetInputPerson():undefined;
		
		if(this.GIP)
			SetPersonXYFloat(this.GIP, -64, -64); //Move out of sight before snapshotting.
		
		RenderMap(); //then render the map to the buffer, but do not call flip screen.
		//SetRenderScript(this.render);
		this.screenOLD = GrabSurface(0, 0, GetScreenWidth(), GetScreenHeight()); //Grab the screen
		//this.updateCurrentInfo(); //Update our current position, done by repos() in warpXYNow().
		this.preWarp=this.preWarp_; this.preWarp_(); //Restore original preWarp()
		
	}

	this.GIP=IsInputAttached()?GetInputPerson():undefined;
	if(this.GIP){
		var GPSS= GetPersonSpriteset(this.GIP);
		var i=GPSS.directions.length-1;
		do{
			if(GPSS.directions[i].name==GetPersonDirection(this.GIP))break;
		}while(i--);
		if(i<0)
			Abort('DEBUG:SpriteSet Corruption'); //direction not found
		this.img = GPSS.images[GPSS.directions[i].frames[GetPersonFrame(this.GIP)].index];
		//Abort(" BASEX1="+GPSS.base.x1+" BASEX2="+GPSS.base.x2+" BASEy1="+GPSS.base.y1+" BASEy2="+GPSS.base.y2)
		this.offsetX=-GPSS.base.x1 - ( (GPSS.base.x2-GPSS.base.x1)>>1 );
		this.offsetY=-GPSS.base.y1 - ( (GPSS.base.y2-GPSS.base.y1)>>1 );
	}


	//.postWarp : Copy new map, then blit them from the far-end, to the startpos, then release. Cleanup preWarp()

	var W = "UpdateMapEngine();"	//Update the mapengine, so our PC is warped to its place (default is centered at entrypoint), 
		+"RenderMap();"		//then render the map to the buffer, but do not call flip screen.
		+"this.screenNEW = GrabSurface(0, 0, GetScreenWidth(), GetScreenHeight());"	//Grab the screen
	;

	if(new_XX === null){	//scroll Vertically
		if(new_YY>=0) { //SOUTH
			W+="var j = GetScreenHeight();var d;"
				+"do{"
					+"this.screenNEW.blit(0,j);"
					+"this.screenOLD.blit(0,j-GetScreenHeight());"
					+"if(this.GIP&&IsPersonVisible(this.GIP))this.img.blitMask(MapToScreenX(this.current.layer,this.current.x)+this.offsetX,0+j+this.offsetY, GetPersonMask(this.GIP));"
					+"this.scrollUpdateAndDraw();"
					+"FlipScreen();"
					;if(this.scrolldelay>0)
					W+="d=GetTime()+this.scrolldelay; while(GetTime()<d){};"
					;if(this.scrollskip>0)
					W+="j-=this.scrollskip";
				W+="}while(--j>0);"
		}else{ //NORTH
			W+="var j = GetScreenHeight();var d;"
				+"do{"
					+"this.screenNEW.blit(0,-j);"
					+"this.screenOLD.blit(0,-j+GetScreenHeight());"
					+"if(this.GIP&&IsPersonVisible(this.GIP))this.img.blitMask(MapToScreenX(this.current.layer,this.current.x)+this.offsetX,GetScreenHeight()-j+this.offsetY, GetPersonMask(this.GIP));"
					+"this.scrollUpdateAndDraw();"
					+"FlipScreen();"
					;if(this.scrolldelay>0)
					W+="d=GetTime()+this.scrolldelay; while(GetTime()<d){};"
					;if(this.scrollskip>0)
					W+="j-=this.scrollskip";
				W+="}while(--j>0);"				
		}
	}else if(new_YY === null){	//Scroll Horizontally
		if(new_XX>=0){	//EAST
			W+="var i = GetScreenWidth();var d;"
				+"do{"
					+"this.screenNEW.blit(i,0);"
					+"this.screenOLD.blit(i-GetScreenWidth(),0);"
					+"if(this.GIP&&IsPersonVisible(this.GIP))this.img.blitMask(i+this.offsetX,MapToScreenY(this.current.layer,this.current.y)+this.offsetY, GetPersonMask(this.GIP));"
					+"this.scrollUpdateAndDraw();"
					+"FlipScreen();"
					;if(this.scrolldelay>0)
					W+="d=GetTime()+this.scrolldelay; while(GetTime()<d){};"
					;if(this.scrollskip>0)
					W+="i-=this.scrollskip";
				W+="}while(--i>0);"
		}else{	//WEST
			W+="var i = GetScreenWidth();var d;"
				+"do{"
					+"this.screenNEW.blit(-i,0);"
					+"this.screenOLD.blit(-i+GetScreenWidth(),0);"
					+"if(this.GIP&&IsPersonVisible(this.GIP))this.img.blitMask(GetScreenWidth()-i+this.offsetX,MapToScreenY(this.current.layer,this.current.y)+this.offsetY, GetPersonMask(this.GIP));"
					+"this.scrollUpdateAndDraw();"
					+"FlipScreen();"
					;if(this.scrolldelay>0)
					W+="d=GetTime()+this.scrolldelay; while(GetTime()<d){};"
					;if(this.scrollskip>0)
					W+="i-=this.scrollskip";
				W+="}while(--i>0);"
		}
	};

	W +=	"this.postWarp_(); this.postWarp=this.postWarp_;" //Run and restore postWarp()
		+"this.active=0;";
	;


	this.postWarp = new Function(W); 


	//now Warp to the other map (disable fadeout/fadein before. Restore it afterwards)
	this.fadeOut_=this.fadeOut;
	this.fadeOut=function(){};
	this.fadeOutFrames_=this.fadeOutFrames;
	this.fadeOutFrames=0;
	this.fadeIn_=this.fadeIn;
	this.fadeIn=function(){};

	this.warpXY(new_map,new_XX,new_YY);

	this.fadeIn=this.fadeIn_;
	this.fadeOut=this.fadeOut_;
	this.fadeOutFrames=this.fadeOutFrames_;
	return 1; //return Success (this.active is still 1)
}

/**
 * Set a fade as default fade for map change transitions
 * @param {string} fade The name of the fade you want to set.
 * @returns true if that fade could be set, false if that fade is not defined.
 */
TeleporthicaEngine.prototype.setFade = function(fade) {
	if(this.FADES[fade]){
		this.fadeIn = this.FADES[fade].fadeIn;
		this.fadeOut= this.FADES[fade].fadeOut;
		this.fadeInFrames = this.FADES[fade].fadeInFrames;
		this.fadeOutFrames= this.FADES[fade].fadeOutFrames;
		this.fadeName=fade;
		this.fadeName_="";
		return true;
	}else
		//return false; //change the abort to this if you want it a little more benign, like when releasing a game...
		Abort("Teleporthica.setFade("+fade+") Cannot find this fade");
}


/**
 * Set a fade only for the next map change transitions
 * @param {string} fade The name of the fade you want to set.
 * @returns true if that fade could be set, false if that fade is not defined.
 */
TeleporthicaEngine.prototype.setNextFade = function(fade) {
	var currentfade = this.fadeName;
	if(!this.setFade(fade)){
		this.fadeName_="";
		return false;
	}else{
		if(!this.fadeName_)
			this.fadeName_ =  currentfade;
		return true;
	}
}


/**
 * Add a new fade to the engine.
 * @param {string} fadename The name of the fade you want to define
 * @param {integer} fadeOutFrames the number of frames you want to start fadeOutFunc() before changing maps
 * @param {integer} fadeInFrames the number of frames you want to let fadeInFunc() run before MapChange.active is false again.
 * @param {function} fadeOutFunc A function that performs a fade out.
 * @param {function} fadeInFunc A function that performs a fade in.
 */
TeleporthicaEngine.prototype.defineFade = function(fadename,fadeOutFrames,fadeInFrames,fadeOutFunc,fadeInFunc) {
	this.FADES[fadename] = {
		fadeOutFrames: fadeOutFrames,
		fadeInFrames: fadeInFrames,
		fadeOut: fadeOutFunc,
		fadeIn: fadeInFunc
	};
}

// --------------- //

MapChange.defineFade('spot', 10,10,
	function(){
		MapChange.SFW = CreateSurface(GetScreenWidth(), GetScreenHeight(), CreateColor(0,0,0,128));
		MapChange.SFC = CreateSurface(16, 16, CreateColor(0,0,0,0));
		MapChange.SFC.rescale(40,40);
		DarkRoomAround(GetInputPerson(),GetScreenWidth(),0,0,0,0)

		MapChange.UpdateAndDraw = function(n){
			MapChange.SFW.blitSurface(MapChange.SFC,40, 40);
			MapChange.SFW.blit(10,10);
			if(--n<1){
				MapChange.SFW = null;
				MapChange.SFC = null;
				return;
			}
		}
	//because its not defined in renderscripts with the other UpdateAndDraw()
	MapChange.UpdateAndDraw(MapChange.FADES['spot'].fadeOutFrames);
	},
	function(){
		SetColorMask(CreateColor(0, 0, 0,   0), this.fadeInFrames, 'spot');
	}
);

// Fadeout to gray, then slowly to coloured new map
MapChange.defineFade('grayfade', 1,40,
	function(){
		var CMgrayscale= CreateColorMatrix(0,  85, 85, 85,    0,  85, 85, 85,    0,  85, 85, 85);
		//Move out of sight before snapshotting.
		if((this.GIP=this.GIP?this.GIP:IsInputAttached()?GetInputPerson():undefined) && (this.v = IsPersonVisible(this.GIP)) ) SetPersonVisible(this.GIP, false);
		RenderMap(); //then render the map to the buffer, but do not call flip screen.
		this.screenOLD = GrabSurface(0, 0, GetScreenWidth(), GetScreenHeight()); //Grab the screen
		this.screenOLD.applyColorFX(0, 0, GetScreenWidth(), GetScreenHeight(), CMgrayscale);
		this.alpha=155;
	},
	function(){
		if(this.v) 
			SetPersonVisible(this.GIP, true);
		delete this.v;
		MapChange.UpdateAndDraw = function(n){
			this.alpha-=5;
			if(this.alpha<1)return this.done();
			this.screenOLD.setAlpha(this.alpha);
			this.screenOLD.blit(0,0);
		}
	}
);

//Fade and scroll old map, reveal new map.
MapChange.defineFade('wipefade', 1,52,
        function(){
                //Move out of sight before snapshotting.
		if((this.GIP=this.GIP?this.GIP:IsInputAttached()?GetInputPerson():undefined) && (this.v = IsPersonVisible(this.GIP)) ) SetPersonVisible(this.GIP, false);
		RenderMap(); //then render the map to the buffer, but do not call flip screen.
                this.screenOLD = GrabSurface(0, 0, GetScreenWidth(), GetScreenHeight()); //Grab the screen
                this.screenOLD = GrabSurface(0, 0, GetScreenWidth(), GetScreenHeight()); //Grab the screen
		this.alpha=256;
		if(Lithonite){
			this.factorX = Lithonite.hist_x;
			this.factorY = Lithonite.hist_y;
		}else{
			if(IsInputAttached){
				var d = GetPersonDirection(GetInputPerson());
				this.factorX = this.factorY = 0; 
				if(d.match(/north/)) this.factorY = -1; 
				if(d.match(/south/)) this.factorY = 1; 
				if(d.match(/west/)) this.factorX = -1; 
				if(d.match(/east/)) this.factorX = 1; 
			}
		}
        },
        function(){
		if(this.v) 
			SetPersonVisible(this.GIP, true);
		delete this.v;
                MapChange.UpdateAndDraw = function(n){
                        this.alpha-=5;
                        if(this.alpha<1)
				return this.done();
                        this.screenOLD.setAlpha(this.alpha);
                        this.screenOLD.blit(
				this.factorX*(255-this.alpha),
				this.factorY*(255-this.alpha)
			);
                }
        }
);

MapChange.defineFade('wipefadeInverted', 1,52,
	function(){
		this.FADES['wipefade'].fadeOut.call(this);
		this.factorX *= -1;
		this.factorY *= -1;
	},
	function(){
		this.FADES['wipefade'].fadeIn.call(this);
	}
);


// Oldmap will be sliced, then pulled both to the right and left
MapChange.defineFade('stripeout', 1,20,
        function(){
		this.screenOLD = GrabSurface(0, 0, GetScreenWidth(), GetScreenHeight()); //Grab the screen
		this.n=GetScreenWidth();
		this.h=Math.floor(GetScreenHeight()/10);
		this.alpha=256;
        },
        function(){
                MapChange.UpdateAndDraw = function(n){
			this.n-=20;
                        if(this.n<0)return this.done();
                        this.alpha-= this.alpha>20?20:1;
                        if(this.alpha<0)this.alpha=0;
                        this.screenOLD.setAlpha(this.alpha);
			var s;
			var i=9;
			do{
				s = this.screenOLD.cloneSection(0, i*this.h, GetScreenWidth(), this.h);
				if(i%2){
                        		s.blit(GetScreenWidth()-this.n,i*this.h);
				}else{
                        		s.blit(this.n-GetScreenWidth(),i*this.h);
				}
			}while(i--);
                }
        }
);

MapChange.defineFade('pixelate', 0,0,
        function(){
		this.fps = GetMapEngineFrameRate();
		SetMapEngineFrameRate(1); //Give us oopf by skipping the drawings...
                //Move out of sight before snapshotting.
		if((this.GIP=this.GIP?this.GIP:IsInputAttached()?GetInputPerson():undefined) && (this.v = IsPersonVisible(this.GIP)) ) SetPersonVisible(this.GIP, false);
                RenderMap(); //then render the map to the buffer, but do not call flip screen.
                this.screenOLD = GrabSurface(0, 0, GetScreenWidth(), GetScreenHeight()); //Grab the screen
		var t; var screen;
		for(var i=1;i<7;++i)
		{
			t=GetTime()+60;
			screen = this.screenOLD.clone();
			screen.rescale(screen.width>>i,screen.height>>i);
			screen.rescale(GetScreenWidth(),GetScreenHeight());
			screen.blit(0,0);
			FlipScreen();
			while(t>GetTime()){};
		}
        },
        function(){
		if(this.v) 
			SetPersonVisible(this.GIP, true);
		delete this.v;
		SetMapEngineFrameRate(1);
		RenderMap();
                this.screenNEW = GrabSurface(0, 0, GetScreenWidth(), GetScreenHeight()); //Grab the screen
		var i=7; var t; var screen;
		do{
			t=GetTime()+60;
			screen = this.screenNEW.clone();
			screen.rescale(screen.width>>i,screen.height>>i);
			screen.rescale(GetScreenWidth(),GetScreenHeight());
			screen.blit(0,0);
			FlipScreen();
			while(t>GetTime()){};
		}while(--i);
		SetMapEngineFrameRate(this.fps||60); //restore FPS
        }
);

MapChange.defineFade('zoomoutin', 0,1,
        function(){
		this.fps = GetMapEngineFrameRate();
		SetMapEngineFrameRate(1); //Give us oopf by skipping the drawings...
		if(!this.zoomPerson) this.zoomPerson=this.GIP?this.GIP:IsInputAttached()?GetInputPerson():undefined;
                RenderMap(); //then render the map to the buffer, but do not call flip screen.
                var screen= GrabImage(0, 0, GetScreenWidth(), GetScreenHeight()); //Grab the screen
		var x=GetScreenWidth()>>1; var y=GetScreenHeight()>>1;
		if(typeof this.zoomPerson=="string"){
			x= MapToScreenX( GetPersonLayer(this.GIP), GetPersonX(this.GIP) );
			y= MapToScreenY( GetPersonLayer(this.GIP), GetPersonY(this.GIP) );
		}else if(typeof this.zoomPerson=="Object"){
			x=this.zoomPerson.x;
			y=this.zoomPerson.y;
		}
		if(x<0)x=0; if(x>GetScreenWidth()) x=GetScreenWidth();
		if(y<0)y=0; if(y>GetScreenHeight()) y=GetScreenHeight();
		var z= 7;
		var i=z; var t;
		do{
			t=GetTime()+60;
			screen.zoomBlit(x-i/z*x,y-i/z*y,i/z);
			FlipScreen();
			while(t>GetTime()){};
		}while(--i);
        },
	function(){
		RenderMap();
                var screen= GrabImage(0, 0, GetScreenWidth(), GetScreenHeight()); // Grab the screen
		ApplyColorMask(CreateColor(0, 0, 0, 255), 0); // ClearScreen()
		if(!this.zoomPerson) this.zoomPerson=this.GIP?this.GIP:IsInputAttached()?GetInputPerson():undefined;
		var x=GetScreenWidth()>>1; var y=GetScreenHeight()>>1;
		if(typeof this.zoomPerson=="string"){
			x= MapToScreenX( GetPersonLayer(this.GIP), GetPersonX(this.GIP) );
			y= MapToScreenY( GetPersonLayer(this.GIP), GetPersonY(this.GIP) );
		}else if(typeof this.zoomPerson=="Object"){
			 x=this.zoomPerson.x;
			 y=this.zoomPerson.y;
		}
		if(x<0)x=0; if(x>GetScreenWidth()) x=GetScreenWidth();
		if(y<0)y=0; if(y>GetScreenHeight()) y=GetScreenHeight();
		var z= 7;
		var i=1; var t;
		do{
			t=GetTime()+60;
			screen.zoomBlit(x-i/z*x,y-i/z*y,i/z);
			FlipScreen();
			while(t>GetTime()){};
		 }while(++i<z);
		SetMapEngineFrameRate(this.fps||60); //restore FPS
	}
);

MapChange.defineFade('zoominfade', 0,40,
	function(){
		this.screenOLD = GrabSurface(0, 0, GetScreenWidth(), GetScreenHeight()); //Grab the screen
		this.n=GetScreenWidth();
		this.h=this.alpha;
		this.alpha=256;
		this.x = GetScreenWidth();
		this.y = GetScreenHeight();
	},
	function(){
		MapChange.UpdateAndDraw = function(n){
			this.alpha-=10;
			if(this.alpha<1)return this.done();
			this.screenOLD.setAlpha(this.alpha);
			this.h = 256 - this.alpha;
			this.screenOLD.createImage().transformBlit(0-this.h,0-this.h, this.x+this.h,0-this.h, this.x+this.h,this.y+this.h, 0-this.h, this.y+this.h);

		}
	}
);


/*
	//Example how to enhance Teleporthica with Neologix's N_trans transitions:

//1. First include the library

RequireScript("/common/neologix/scripts/n_trans.js");


//2. Define some transitions using defineFade()

MapChange.defineFade('neologix_zoom', 1,1,
	//We zomeIn as fadeout :)
	function(){ tZoomIn(2, 500, -1, -1); },
	function(){ tZoomOut(1, 500, -1, -1); }
);

MapChange.defineFade('neologix_pixelate', 1,1,
        function(){ tPixelIn(1,1000); },
        function(){ tPixelOut(1,1000); }
);

MapChange.defineFade('neologix_swish', 1,1,
        function(){ tSwish(64, 1000, CreateColor(0,0,0,0)); }
        function(){ tSwish(-64, 1000, CreateColor(0,0,0,255)); }
);

MapChange.defineFade('neologix_melt_down', 1,1,
        function(){
                this.screenOLD = GrabSurface(0, 0, GetScreenWidth(), GetScreenHeight());
        },
        function(){
		this.screenNEW = GrabSurface(0, 0, GetScreenWidth(), GetScreenHeight());
		tMelt(2, 5, 4, CreateColor(0,0,0,255), 1, this.screenOLD ,this.screenNEW);
		//tMelt(4, 5, 4, CreateColor(0,0,0,255), 5, this.screenOLD ,this.screenNEW); //Inverted V shape up

        }
);

MapChange.defineFade('neologix_split', 1,1,
        function(){
                this.screenOLD = GrabSurface(0, 0, GetScreenWidth(), GetScreenHeight());
        },
        function(){
                this.screenNEW = GrabSurface(0, 0, GetScreenWidth(), GetScreenHeight());
		tSplit(4, 5, 4, CreateColor(0,0,0,155), 5, this.screenOLD,this.screenNEW);
		//tSplit(4, 5, 4, CreateColor(0,0,0,155), 5); //this looks nice too

        }
);

//This one is very slow
MapChange.defineFade('neologix_iris', 1,1,
	function(){ transition('iris', 'In', 100, CreateColor(0,0,0, 255));},
	function(){ transition('iris', 'Out', 100, CreateColor(0,0,0, 255));}
);



//Another one
MapChange.defineFade('neologix_swish', 1,1,
	function(){
		tSwish(64, 500, CreateColor(200,100,0,150));
	},
	function(){
		RenderMap();
		tSwish(-64, 500, CreateColor(200,100,0,150));
	}
);
//This function is missing from the old libs, lets define it ourselves:
function deg2rad(d){return d/360*2*3.1415}
//or even:
function deg2rad(d){return d*0.0174532925199433}

//transition() is broken. Usually just because a variable doesnt exist. Will resolve with Neologix
MapChange.defineFade('neologix_broken', 1,1,
        function(){
                transition("wipeVU", "Out", 1000, CreateColor(200,100,0,150))
        },
        function(){
                RenderMap(); 
                transition("wipeVU", "In", 1000, CreateColor(200,100,0,150))
        }
//      function(){tPixelOut(1,50)}
);
//In older versions of the libs, you need to define this functions:
function ScrShot(){return ScrToImage();}



//Now activate one of the transitions:
MapChange.setFade('neologix_swish');


*/

